<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Testimonials Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Testimonials_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['mwt-testimonials-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['mwt-testimonials-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-testimonials-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Testimonials Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-testimonial-carousel';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'testimonials', 'carousel', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Carousel Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'testimonials_layout',
			[
				'label'       => __('Choose testimonials layout', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SELECT,
				'options'     => [
					'default' => __('Default', 'mwt-addons-for-elementor'),
					'second_layout'  => __('Second', 'mwt-addons-for-elementor'),
					'third_layout'  => __('Third', 'mwt-addons-for-elementor'),
				],
				'default'     => 'default',
				'separator'   => 'after',
				'label_block' => true,
			]
		);

		/* Start repeater */
		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'item_name',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'description' => __('Enter the Name of the Person to testimonial', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_position',
			[
				'label'       => __('Position', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'description' => __('Can be used for a job description', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_testimonial',
			[
				'label'       => __('Testimonial', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_testimonial_date',
			[
				'label' => esc_html__('Date Of Testimonial', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::DATE_TIME,
				'picker_options' => [
					'enableTime' => false,
					'dateFormat' => "Y.m.d",
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'testimonials-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'image' => '',
						'item_name' => esc_html__('Name 1', 'mwt-addons-for-elementor'),
						'item_position' => esc_html__('Position 1', 'mwt-addons-for-elementor'),
						'item_testimonial' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_testimonial_date' => '',
					],
					[
						'image' => '',
						'item_name' => esc_html__('Name 2', 'mwt-addons-for-elementor'),
						'item_position' => esc_html__('Position 2', 'mwt-addons-for-elementor'),
						'item_testimonial' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_testimonial_date' => '',
					],
					[
						'image' => '',
						'item_name' => esc_html__('Name 3', 'mwt-addons-for-elementor'),
						'item_position' => esc_html__('Position 3', 'mwt-addons-for-elementor'),
						'item_testimonial' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_testimonial_date' => '',
					],
				],
				'title_field' => '{{{ item_name }}}',
			]
		);

		$this->add_control(
			'quote_icon',
			[
				'label'            => __('Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-quote-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'quote-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'data_rtl',
			[
				'label'   => __('Enable RTL', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'min' => 0,
				'step' => 50,
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 4,
				'separator' => 'before',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'data_responsive_md',
			[
				'label' => esc_html__('Laptop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tablet Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 80,
				'step' => 1,
				'default' => 30,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav_prev_next',
			[
				'label'   => __('Show prev/next', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'date_nav_prev',
			[
				'label'            => __('Previous Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_prev_word',
			[
				'label'     => __('Previous word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Prev', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'date_nav_next',
			[
				'label'            => __('Next Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_next_word',
			[
				'label'     => __('Next word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Next', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'data_nav_dots',
			[
				'label'   => __('Show dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'testimonials_layout!' => 'second_layout',
				],
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'testimonials_item_align',
			[
				'label'     => __('Align Items', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Start', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-start-h',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-center-h',
					],
					'end'   => [
						'title' => __('End', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-end-h',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item' => 'align-items: {{VALUE}}; text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'testimonials_item_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item',
				'condition' => [
					'testimonials_layout!' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'testimonials_item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout!' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'testimonials_top_block_options',
			[
				'label' => esc_html__('Top Block', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'top_block_align',
			[
				'label'     => __('Align', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Top', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-start-v',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-center-v',
					],
					'end'   => [
						'title' => __('Bottom', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-end-v',
					],
				],
				'default'   => 'start',
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap' => 'align-items: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'top_block_gap',
			[
				'label'      => __('Gap', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap' => 'gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'top_block_height',
			[
				'label'      => __('Height (px)', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 90,
				],
				'selectors'  => [
					'{{WRAPPER}} .third-layout .owl-item:not(.active)+.owl-item.active .mwt-testimonials-carousel-image-wrap' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);


		$this->add_responsive_control(
			'top_block_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .third-layout .owl-item:not(.active)+.owl-item.active .mwt-testimonials-carousel-image-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'testimonials_bottom_block_options',
			[
				'label' => esc_html__('Bottom Block', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'item_bottom_block_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .mwt-testimonials-carousel-content-wrap' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .third-layout .mwt-testimonials-carousel-content-wrap::before' => 'border-bottom-color: {VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);


		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'testimonials_item_bottom_block_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .mwt-testimonials-carousel-content-wrap',
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'testimonials_item_bottom_block_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .mwt-testimonials-carousel-content-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'item_bottom_block_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .mwt-testimonials-carousel-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'item_bottom_block_height_author_info',
			[
				'label'      => __('Height Author Info (px)', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 58,
				],
				'selectors'  => [
					'{{WRAPPER}} .third-layout .mwt-testimonials-carousel-name-position-wrap' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'testimonials_active_item_options',
			[
				'label' => esc_html__('Bottom Block Active', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'active_item_bottom_block_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .third-layout .owl-item:not(.active)+.owl-item.active .mwt-testimonials-carousel-content-wrap .mwt-testimonials-carousel-testimonial' => 'color: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'active_item_bottom_block_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .third-layout .owl-item:not(.active)+.owl-item.active .mwt-testimonials-carousel-content-wrap' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .third-layout .mwt-testimonials-carousel-content-wrap::before' => 'border-bottom-color: {VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'item_bottom_block_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .mwt-testimonials-carousel-content-wrap' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .third-layout .mwt-testimonials-carousel-content-wrap::before' => 'border-bottom-color: {VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'third_layout',
				]
			]
		);

		$this->add_control(
			'center_item_options',
			[
				'label' => esc_html__('Item Center', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'data_center' => 'yes',
				]
			]
		);

		$this->add_control(
			'center_item_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-item.center .owl-carousel-item' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'data_center' => 'yes',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'testimonials_center_item_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .owl-item.center .owl-carousel-item',
				'condition' => [
					'data_center' => 'yes',
				]
			]
		);

		$this->add_control(
			'testimonials_center_item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-item.center .owl-carousel-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'data_center' => 'yes',
				]
			]
		);

		$this->add_control(
			'divider',
			[
				'label' => esc_html__('Divider', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'divider_width',
			[
				'label'      => __('Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-divider'  => 'width: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'divider' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'divider_weight',
			[
				'label'      => __('Weight', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-divider'  => 'height: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'divider' => 'yes',
				],
			]
		);

		$this->add_control(
			'divider_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-divider' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'divider' => 'yes',
				],
			]
		);

		$this->add_control(
			'item_elements_order',
			[
				'label' => esc_html__('Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'item_image_order',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 5,
				'step' => 1,
				'default' => 1,
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap' => 'order: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'item_name_order',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 5,
				'step' => 1,
				'default' => 2,
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-name' => 'order: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'item_position_order',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 5,
				'step' => 1,
				'default' => 3,
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-position' => 'order: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->add_control(
			'item_divider_order',
			[
				'label' => esc_html__('Divider', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 5,
				'step' => 1,
				'default' => 4,
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-divider' => 'order: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
					'divider' => 'yes',
				]
			]
		);

		$this->add_control(
			'item_testimonial_order',
			[
				'label' => esc_html__('Testimonial', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'description' => esc_html__('Sorted in ascending order of value.', 'mwt-addons-for-elementor'),
				'min' => 1,
				'max' => 5,
				'step' => 1,
				'default' => 5,
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial' => 'order: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout!' => ['second_layout', 'third_layout'],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image_section',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'image_size',
			[
				'label'      => __('Image Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 50,
						'max'  => 300,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 270,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'image_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-image',
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'image_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout!' => 'third_layout',
				],
			]
		);

		$this->add_control(
			'quote_icon_options',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_responsive_control(
			'quote_icon_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon i'  => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon svg' => 'width: {{SIZE}}{{UNIT}}; heigth: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'quote_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'quote_icon_background',
				'types' => ['classic', 'gradient'],
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon',
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'quote_icon_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon',
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'quote_icon_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'quote_icon_padding',
			[
				'label' => esc_html__('Padding', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'default' => [
					'top' => 9,
					'right' => 9,
					'bottom' => 9,
					'left' => 9,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'quote_icon_position',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom-right',
				'options' => [
					'top-left' => esc_html__('Top Left', 'mwt-addons-for-elementor'),
					'top-right' => esc_html__('Top Right', 'mwt-addons-for-elementor'),
					'bottom-left'  => esc_html__('Bottom Left', 'mwt-addons-for-elementor'),
					'bottom-right' => esc_html__('Bottom Right', 'mwt-addons-for-elementor'),
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'quote_icon_position_top',
			[
				'label' => esc_html__('Top', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'top: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'quote_icon_position' => ['top-left', 'top-right'],
				]
			]
		);

		$this->add_control(
			'quote_icon_position_bottom',
			[
				'label' => esc_html__('Bottom', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'quote_icon_position' => ['bottom-left', 'bottom-right'],
				]
			]
		);

		$this->add_control(
			'quote_icon_position_left',
			[
				'label' => esc_html__('Left', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'quote_icon_position' => ['top-left', 'bottom-left'],
				]
			]
		);

		$this->add_control(
			'quote_icon_position_right',
			[
				'label' => esc_html__('Right', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-image-wrap .mwt-testimonials-carousel-quote-icon' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'quote_icon_position' => ['top-right', 'bottom-right'],
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_name_section',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'name_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-name' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'name_typography',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-name',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'name_shadow',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-name',
			]
		);

		$this->add_responsive_control(
			'name_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_position_section',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'position_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-position' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'position_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-testimonials-carousel-position',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'position_shadow',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-position',
			]
		);

		$this->add_responsive_control(
			'position_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-position' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_testimonial_section',
			[
				'label' => esc_html__('Testimonial', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'testimonial_align',
			[
				'label'     => __('Align', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Start', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-start-h',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-center-h',
					],
					'end'   => [
						'title' => __('End', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-end-h',
					],
				],
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-carousel-item .mwt-testimonials-carousel-testimonial' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'testimonials_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'testimonials_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-testimonials-carousel-testimonial',
			]
		);

		$this->add_control(
			'testimonials_max_width',
			[
				'label' => esc_html__('Max Width', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'testimonials_date_options',
			[
				'label' => esc_html__('Date', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'testimonials_date_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial-date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'testimonials_date_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-testimonials-carousel-testimonial-date',
			]
		);

		$this->add_responsive_control(
			'testimonials_date_margin_top',
			[
				'label'      => __('Top Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial-date'  => 'margin-top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'testimonials_layout!' => 'third_layout',
				]
			]
		);

		$this->add_responsive_control(
			'testimonials_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
				'allowed_dimensions' => 'vertical',
				'separator' => 'before',
				'placeholder' => [
					'top' => '',
					'right' => 'auto',
					'bottom' => '',
					'left' => 'auto',
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'testimonial_quote_icon_options',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_responsive_control(
			'testimonial_quote_icon_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon i'  => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon svg' => 'width: {{SIZE}}{{UNIT}}; heigth: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_position',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom-right',
				'options' => [
					'top-left' => esc_html__('Top Left', 'mwt-addons-for-elementor'),
					'top-right' => esc_html__('Top Right', 'mwt-addons-for-elementor'),
					'bottom-left'  => esc_html__('Bottom Left', 'mwt-addons-for-elementor'),
					'bottom-right' => esc_html__('Bottom Right', 'mwt-addons-for-elementor'),
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_position_top',
			[
				'label' => esc_html__('Top', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon' => 'top: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'testimonial_quote_icon_position' => ['top-left', 'top-right'],
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_position_bottom',
			[
				'label' => esc_html__('Bottom', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon' => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'testimonial_quote_icon_position' => ['bottom-left', 'bottom-right'],
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_position_left',
			[
				'label' => esc_html__('Left', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'testimonial_quote_icon_position' => ['top-left', 'bottom-left'],
				]
			]
		);

		$this->add_control(
			'testimonial_quote_icon_position_right',
			[
				'label' => esc_html__('Right', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-testimonial .mwt-testimonials-carousel-quote-icon' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'testimonials_layout' => 'second_layout',
					'testimonial_quote_icon_position' => ['top-right', 'bottom-right'],
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_prev_next_section',
			[
				'label' => esc_html__('Prev/Next', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'prev_next_icons_position',
			[
				'label' => esc_html__('Prev/Next Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom',
				'options' => [
					'bottom' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'middle' => esc_html__('Middle', 'mwt-addons-for-elementor'),
					'top' => esc_html__('Top', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'prev_next_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__('Start', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-start-h',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-center-h',
					],
					'end' => [
						'title' => esc_html__('End', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-end-h',
					],
					'space-between' => [
						'title' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-justify-space-between-h',
					],
				],
				'default' => 'start',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav' => 'justify-content: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_offset',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav' => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}}; width: calc(100% + {{SIZE}}{{UNIT}} * 2);',
				],
			]
		);

		$this->add_control(
			'hr',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'prev_next_typography',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button',
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 4,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button svg' => 'font-size: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_control(
			'prev_next_icon_position',
			[
				'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'row' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'row-reverse' => esc_html__('Reverse', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button>span' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_spacing',
			[
				'label'      => __('Icon Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 12,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button>span' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'prev_next_style_tabs'
		);

		$this->start_controls_tab(
			'prev_next_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_disabled_tab',
			[
				'label' => esc_html__('Disabled', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_disabled',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_disabled',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_disabled',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_disabled',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button.disabled' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'prev_next_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button',
			]
		);

		$this->add_control(
			'prev_next_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_btn_gap',
			[
				'label'      => __('Buttons Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-nav' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_dots' => 'yes',
				]
			]
		);

		$this->start_controls_tabs(
			'dots_style_tabs'
		);

		$this->start_controls_tab(
			'dots_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dots_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dats_style_active_tab',
			[
				'label' => esc_html__('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel-container .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$unique_id = uniqid();

		if (empty($settings['testimonials-list'])) {
			return;
		} ?>
		<div class="mwt-testimonials-carousel-container layout-gap-30 ">
			<?php
			$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
			$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
			$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
			$data_nav = ($settings['data_nav_prev_next'] === 'yes') ? 'true' : 'false';
			$data_dots = ($settings['data_nav_dots'] === 'yes') ? 'true' : 'false';
			$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
			$data_rtl = ($settings['data_rtl'] === 'yes') ? 'true' : 'false';

			$item_layout = $settings['testimonials_layout'];
			switch ($item_layout) {
				case 'default': ?>
					<div class="owl-carousel mwt-testimonials-carousel" data-number="<?php echo esc_attr($unique_id); ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-md="<?php echo esc_attr($settings['data_responsive_md']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-testimonials-carousel-image-wrap">
										<span>
											<?php echo wp_kses_post($image_html); ?>
										</span>
									</div>
								<?php
								}
								$name = $settings['testimonials-list'][$index]['item_name'];
								$position = $settings['testimonials-list'][$index]['item_position'];
								$testimonial = $settings['testimonials-list'][$index]['item_testimonial'];
								$date = $settings['testimonials-list'][$index]['item_testimonial_date'];
								if (!empty($item['item_name'])) : ?>
									<div class="mwt-testimonials-carousel-name">
										<?php echo esc_html($name); ?>
									</div>
								<?php endif;
								if (!empty($item['item_position'])) : ?>
									<div class="mwt-testimonials-carousel-position">
										<?php echo esc_html($position); ?>
									</div>
								<?php endif;
								if ($settings['divider'] === 'yes') : ?>
									<div class="mwt-testimonials-carousel-divider"></div>
								<?php endif;
								if (!empty($item['item_testimonial'])) : ?>
									<div class="mwt-testimonials-carousel-testimonial">
										<?php echo esc_html($testimonial);
										if (!empty($date)) : ?>
											<div class="mwt-testimonials-carousel-testimonial-date"><?php echo esc_html($date); ?></div>
										<?php endif; ?>
									</div>
								<?php endif; ?>
							</div>
						<?php } ?>
					</div>
				<?php break;

				case 'second_layout': ?>
					<div class="grid-wrapper cols-2">
						<div class="grid-item">
							<div class="owl-carousel mwt-testimonials-carousel-sync1" data-margin="<?php echo esc_attr($settings['data_margin']) ?>">
								<?php foreach ($settings['testimonials-list'] as $index => $item) { ?>
									<div class="owl-carousel-item">
										<?php
										$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
										if (!$image_url && isset($item['image']['url'])) {
											$image_url = $item['image']['url'];
										}
										if (!empty($item['item_name'])) {
											$image_alt = $item['item_name'];
										} else {
											$image_alt = 'This is a name';
										}
										$image_html = '<img class="mwt-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
										if (!empty($item['image']['url'])) {
										?>
											<div class="mwt-testimonials-carousel-image-wrap">
												<?php echo wp_kses_post($image_html);
												$quote_icon = isset($settings['quote_icon']) ? $settings['quote_icon'] : '';
												if ($quote_icon) { ?>
													<div class="mwt-testimonials-carousel-quote-icon"><?php \Elementor\Icons_Manager::render_icon($quote_icon, ['aria-hidden' => 'true']); ?></div>
												<?php } ?>
											</div>
										<?php
										}
										$name = $settings['testimonials-list'][$index]['item_name'];
										$position = $settings['testimonials-list'][$index]['item_position'];
										if (!empty($item['item_name'])) : ?>
											<div class="mwt-testimonials-carousel-name">
												<?php echo esc_html($name); ?>
											</div>
										<?php endif;
										if (!empty($item['item_position'])) : ?>
											<div class="mwt-testimonials-carousel-position">
												<?php echo esc_html($position); ?>
											</div>
										<?php endif; ?>
									</div>
								<?php } ?>
							</div>
						</div>
						<div class="grid-item">
							<div class="owl-carousel mwt-testimonials-carousel-sync2">
								<?php
								foreach ($settings['testimonials-list'] as $index => $item) { ?>
									<div class="owl-carousel-item">
										<?php
										$testimonial = $settings['testimonials-list'][$index]['item_testimonial'];
										$date = $settings['testimonials-list'][$index]['item_testimonial_date'];
										if ($settings['divider'] === 'yes') : ?>
											<div class="mwt-testimonials-carousel-divider"></div>
										<?php endif;
										if (!empty($item['item_testimonial'])) : ?>
											<div class="mwt-testimonials-carousel-testimonial">
												<?php
												$quote_icon = isset($settings['quote_icon']) ? $settings['quote_icon'] : '';
												if ($quote_icon) { ?>
													<div class="mwt-testimonials-carousel-quote-icon"><?php \Elementor\Icons_Manager::render_icon($quote_icon, ['aria-hidden' => 'true']); ?></div>
												<?php }
												echo esc_html($testimonial);
												if (!empty($date)) : ?>
													<div class="mwt-testimonials-carousel-testimonial-date"><?php echo esc_html($date); ?></div>
												<?php endif; ?>
											</div>
										<?php endif; ?>
									</div>
								<?php } ?>

							</div>
						</div>
					</div>
				<?php break;

				case 'third_layout': ?>
					<div class="owl-carousel mwt-testimonials-carousel third-layout" data-nav-speed="500" data-number="<?php echo esc_attr($unique_id); ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-loop="true" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="false" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="2" data-responsive-md="2" data-responsive-sm="2" data-responsive-xs="1">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';

								$name = $settings['testimonials-list'][$index]['item_name'];
								$position = $settings['testimonials-list'][$index]['item_position'];
								$testimonial = $settings['testimonials-list'][$index]['item_testimonial'];
								$date = $settings['testimonials-list'][$index]['item_testimonial_date'];

								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-testimonials-carousel-image-wrap">
										<?php echo wp_kses_post($image_html); ?>
										<div class="mwt-testimonials-carousel-top-quote-block">
											<?php if (!empty($item['item_name'])) : ?>
												<div class="mwt-testimonials-carousel-name">
													<?php echo esc_html($name); ?>
												</div>
											<?php endif;
											if (!empty($item['item_position'])) : ?>
												<div class="mwt-testimonials-carousel-position">
													<?php echo esc_html($position);
													if (!empty($date)) : ?>, <span class="mwt-testimonials-carousel-testimonial-date"><?php echo esc_html($date); ?></span>
												<?php endif; ?>
												</div>
											<?php endif; ?>
										</div>
									</div>
								<?php
								} ?>
								<div class="mwt-testimonials-carousel-content-wrap">
									<?php
									if (!empty($item['item_testimonial'])) : ?>
										<div class="mwt-testimonials-carousel-testimonial">
											<?php echo esc_html($testimonial); ?>
										</div>
									<?php endif; ?>
									<?php if ($settings['divider'] === 'yes') : ?>
										<div class="mwt-testimonials-carousel-divider"></div>
									<?php endif; ?>
									<div class="mwt-testimonials-carousel-name-position-wrap">
										<?php
										if (!empty($item['item_name'])) : ?>
											<div class="mwt-testimonials-carousel-name">
												<?php echo esc_html($name); ?>
											</div>
										<?php endif;
										if (!empty($item['item_position'])) : ?>
											<div class="mwt-testimonials-carousel-position">
												<?php echo esc_html($position);
												if (!empty($date)) : ?>, <span class="mwt-testimonials-carousel-testimonial-date"><?php echo esc_html($date); ?></span>
											<?php endif; ?>
											</div>
										<?php endif; ?>
									</div>
								</div>
							</div>
						<?php } ?>
					</div>
			<?php break;
			} ?>
			<div class="owl-nav position-<?php echo esc_attr($settings['prev_next_icons_position']); ?> mwt-testimonials-carousel-nav-<?php echo esc_attr($unique_id); ?>">
				<?php
				$date_nav_prev = isset($settings['date_nav_prev']) ? $settings['date_nav_prev'] : '';
				$date_nav_next = isset($settings['date_nav_next']) ? $settings['date_nav_next'] : '';
				$date_nav_prev_word = isset($settings['date_nav_prev_word']) ? $settings['date_nav_prev_word'] : '';
				$date_nav_next_word = isset($settings['date_nav_next_word']) ? $settings['date_nav_next_word'] : '';
				?>
				<span class="mwt-testimonials-carousel-prev-<?php echo esc_attr($unique_id); ?>">
					<?php if ($date_nav_prev) :
						\Elementor\Icons_Manager::render_icon($settings['date_nav_prev'], ['aria-hidden' => 'true', 'class' => 'icon-prev']);
					endif;
					echo esc_html($date_nav_prev_word)
					?>
				</span>
				<span class="mwt-testimonials-carousel-next-<?php echo esc_attr($unique_id); ?>">
					<?php
					echo esc_html($date_nav_next_word);
					if ($date_nav_next) :
						\Elementor\Icons_Manager::render_icon($settings['date_nav_next'], ['aria-hidden' => 'true', 'class' => 'icon-next']);
					endif;
					?>
				</span>
			</div>
		</div>
<?php }
}
