<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Services Grid Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Services_Grid extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['mwt-services-grid-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['mwt-services-grid-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-services-grid';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Services', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image grid widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-settings';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'services', 'grid'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'services_layout',
			[
				'label' => esc_html__('Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'grid',
				'options' => [
					'grid' => esc_html__('Masonry Grid', 'mwt-addons-for-elementor'),
					'carousel' => esc_html__('Carousel', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'services_item_style',
			[
				'label' => esc_html__('Choose Item Style', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'second_layout' => esc_html__('Second', 'mwt-addons-for-elementor'),
					'third_layout' => esc_html__('Third', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		/* Start repeater */

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'services_bg_image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'services_image_entire_item',
			[
				'label' => esc_html__('Image on the entire item', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$repeater->add_control(
			'services_item_icon',
			[
				'label'     => __('Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default'   => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-cogs',
				],
				'recommended' => [
					'fa-solid' => [
						'cogs',
					],
				],
			]
		);

		$repeater->add_control(
			'services_item_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'services_item_title_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'services_item_category',
			[
				'label'       => __('Category', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'description' => __('To assign for multiple categories, separate by a comma \',\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'services_item_description',
			[
				'label'       => __('Description', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'services_item_button',
			[
				'label' => esc_html__('Button Text', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Type your text here', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Read More', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$repeater->add_control(
			'services_item_button_link',
			[
				'label'       => __('Button Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'services-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'services_bg_image' => '',
						'services_image_entire_item' => '',
						'services_item_icon' => '',
						'services_item_title' => esc_html__('Title 1', 'mwt-addons-for-elementor'),
						'services_item_title_link' => '',
						'services_item_category' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'services_item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
						'services_item_button' => esc_html__('Read More', 'mwt-addons-for-elementor'),
						'services_item_button_link' => '',
					],
					[
						'services_bg_image' => '',
						'services_image_entire_item' => '',
						'services_item_icon' => '',
						'services_item_title' => esc_html__('Title 2', 'mwt-addons-for-elementor'),
						'services_item_title_link' => '',
						'services_item_category' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'services_item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
						'services_item_button' => esc_html__('Read More', 'mwt-addons-for-elementor'),
						'services_item_button_link' => '',
					],
					[
						'services_bg_image' => '',
						'services_image_entire_item' => '',
						'services_item_icon' => '',
						'services_item_title' => esc_html__('Title 3', 'mwt-addons-for-elementor'),
						'services_item_title_link' => '',
						'services_item_category' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'services_item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
						'services_item_button' => esc_html__('Read More', 'mwt-addons-for-elementor'),
						'services_item_button_link' => '',
					],
				],
				'title_field' => '{{{ services_item_title }}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'default' => 'full',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'services_categories',
			[
				'label' => __('Categories', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'services_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'services_filter' => 'yes',
				],
			]
		);

		$category_repeater = new \Elementor\REPEATER();

		$category_repeater->add_control(
			'services_cat',
			[
				'label'   => __('Category', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$category_repeater->add_control(
			'services_cat_link',
			[
				'label'       => __('Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'services_cats_content',
			[
				'label'       => __('Categories', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $category_repeater->get_controls(),
				'default'     => [
					[
						'services_cat' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'services_cat_link' => '',
					],
					[
						'services_cat' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'services_cat_link' => '',
					],
					[
						'services_cat' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'services_cat_link' => '',
					],
				],
				'title_field' => '{{{ services_cat }}}',
				'condition'   => [
					'services_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'services_filters_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .filters' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'services_filter' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'grid_settings',
			[
				'label' => __('Grid Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'services_layout' => 'grid',
				]
			]
		);

		$this->add_control(
			'items_per_row',
			[
				'label' => esc_html__('Items per row', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'description' => esc_html__('How many items should be shown per row?', 'mwt-addons-for-elementor'),
				'min'  => 1,
				'max'  => 4,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'layout_gap',
			[
				'label' => esc_html__('Gap', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => '30',
				'options' => [
					'30' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'0' => esc_html__('None', 'mwt-addons-for-elementor'),
					'1'  => esc_html__('1px', 'mwt-addons-for-elementor'),
					'2' => esc_html__('2px', 'mwt-addons-for-elementor'),
					'3' => esc_html__('3px', 'mwt-addons-for-elementor'),
					'4' => esc_html__('4px', 'mwt-addons-for-elementor'),
					'5' => esc_html__('5px', 'mwt-addons-for-elementor'),
					'10' => esc_html__('10px', 'mwt-addons-for-elementor'),
					'15' => esc_html__('15px', 'mwt-addons-for-elementor'),
					'20' => esc_html__('20px', 'mwt-addons-for-elementor'),
					'30' => esc_html__('30px', 'mwt-addons-for-elementor'),
					'40' => esc_html__('40px', 'mwt-addons-for-elementor'),
					'50' => esc_html__('50px', 'mwt-addons-for-elementor'),
					'60' => esc_html__('60px', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'horizontal_order',
			[
				'label' => esc_html__('Horizontal Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'services_layout' => 'carousel',
				]
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'services_item_style!' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'data_rtl',
			[
				'label'   => __('Enable RTL', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'min' => 0,
				'step' => 50,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'services_item_style!' => 'second_layout',
				],
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 4,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_md',
			[
				'label' => esc_html__('Laptop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tablet Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 80,
				'step' => 1,
				'default' => 30,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav_prev_next',
			[
				'label'   => __('Show prev/next', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'date_nav_prev',
			[
				'label'            => __('Previous Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_prev_word',
			[
				'label'     => __('Previous word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Prev', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'date_nav_next',
			[
				'label'            => __('Next Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_next_word',
			[
				'label'     => __('Next word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Next', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'data_nav_dots',
			[
				'label'   => __('Show dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_grid_item',
			[
				'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'grid_item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'end'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'start',
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap' => 'align-items: {{VALUE}}; text-align: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap' => 'align-items: {{VALUE}}; text-align: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'grid_item_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap, .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap',
			]
		);

		$this->add_control(
			'grid_item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'item_image_options',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'item_image_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-image-wrap',
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'item_image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-image-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'item_content_options',
			[
				'label' => esc_html__('Content', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'item_content_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-content-wrap' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'item_content_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-content-wrap',
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_control(
			'item_content_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-content-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
				],
			]
		);

		$this->add_responsive_control(
			'grid_item_padding',
			[
				'label'      => __('Content Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'grid_item_margin',
			[
				'label'      => __('Content Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_content_gap',
			[
				'label'      => __('Gap', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .second_layout .mwt-services-grid-item-wrap'  => 'gap: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
				]
			]
		);

		$this->start_controls_tabs(
			'grid_item_style_tabs'
		);

		$this->start_controls_tab(
			'grid_item_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'grid_item_overlay',
			[
				'label' => esc_html__('Overlay', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item.second_layout .mwt-services-grid-image-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item.second_layout .mwt-services-grid-image-wrap::before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'grid_item_overlay_opacity',
			[
				'label'      => __('Opacity', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					[
						'min' => 0.1,
						'max' => 1,
						'step' => 0.01,
					]
				],
				'default' => [
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item.second_layout .mwt-services-grid-image-wrap::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item.second_layout .mwt-services-grid-image-wrap::before'  => 'opacity: {{SIZE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'grid_item_box_shadow',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap, .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'grid_item_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'grid_item_hover_overlay',
			[
				'label' => esc_html__('Overlay', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap:hover::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap:hover::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item.second_layout .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item.second_layout .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap::before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'grid_item_hover_overlay_opacity',
			[
				'label'      => __('Opacity', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					[
						'min' => 0.1,
						'max' => 1,
						'step' => 0.01,
					]
				],
				'default' => [
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap:hover::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap:hover::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item.second_layout .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item.second_layout .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap::before'  => 'opacity: {{SIZE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'grid_item_hover_box_shadow',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-item-wrap:hover',
				'selector' => '{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-item-wrap:hover',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'show_divider',
			[
				'label'   => __('Show Divider', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'divider_width',
			[
				'label'      => __('Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-divider'  => 'width: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'show_divider' => 'yes'
				]
			]
		);

		$this->add_responsive_control(
			'divider_weight',
			[
				'label'      => __('Weight', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-divider'  => 'height: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'show_divider' => 'yes'
				]
			]
		);

		$this->add_control(
			'divider_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-divider' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_divider' => 'yes'
				]
			]
		);

		$this->add_responsive_control(
			'divider_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-divider'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'show_divider' => 'yes'
				]
			]
		);

		$this->add_control(
			'show_corners',
			[
				'label'   => __('Show Corners', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'corners_width',
			[
				'label'      => __('Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap.with-corners::before'  => 'border-width: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap.with-corners::after'  => 'border-width: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap.with-corners::before'  => 'border-width: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap.with-corners::after'  => 'border-width: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'show_corners' => 'yes'
				]
			]
		);

		$this->add_control(
			'corners_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap.with-corners::before' => 'border-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-container .grid-item .mwt-services-grid-content-wrap.with-corners::after' => 'border-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap.with-corners::before' => 'border-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-carousel-item .mwt-services-grid-content-wrap.with-corners::after' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'show_corners' => 'yes'
				]
			]
		);

		$this->add_control(
			'show_top_border_hover',
			[
				'label'   => __('Show Top Border Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'top_border_hover_height',
			[
				'label'      => __('Height (px)', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 2,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap.with-top-border-hover::after'  => 'height: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'show_top_border_hover' => 'yes'
				]
			]
		);

		$this->add_control(
			'top_border_hover_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap.with-top-border-hover::after' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_top_border_hover' => 'yes'
				]
			]
		);

		$this->add_control(
			'show_service_counter',
			[
				'label'   => __('Show Service Counter', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
				'condition' => [
					'services_item_style' => 'second_layout',
				]
			]
		);

		$this->add_control(
			'service_counter_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-image-wrap:after' => 'color: {{VALUE}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'service_counter_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-services-grid-image-wrap:after',
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->start_controls_tabs(
			'service_counter_style_tabs'
		);

		$this->start_controls_tab(
			'service_counter_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'service_counter_position_x',
			[
				'label'      => esc_html__('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-image-wrap:after'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'service_counter_position_y',
			[
				'label'      => esc_html__('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-image-wrap:after'  => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_control(
			'service_counter_image_entire_item_options',
			[
				'label' => esc_html__('Image on the entire item', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'service_image_entire_item_counter_position_x',
			[
				'label'      => esc_html__('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .item-entire-image .mwt-services-grid-image-wrap:after'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'service_image_entire_item_counter_position_y',
			[
				'label'      => esc_html__('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .item-entire-image .mwt-services-grid-image-wrap:after'  => 'top: {{SIZE}}{{UNIT}}; bottom:auto;',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'service_counter_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'service_counter_hover_position_x',
			[
				'label'      => esc_html__('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap:after'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'service_counter_hover_position_y',
			[
				'label'      => esc_html__('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap:hover .mwt-services-grid-image-wrap:after'  => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes'
				],
			]
		);

		$this->add_control(
			'service_counter_image_entire_item_options_hover',
			[
				'label' => esc_html__('Image on the entire item', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'service_image_entire_item_counter_hover_position_x',
			[
				'label'      => esc_html__('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap.item-entire-image:hover .mwt-services-grid-image-wrap:after'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'service_image_entire_item_counter_hover_position_y',
			[
				'label'      => esc_html__('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 500,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-item-wrap.item-entire-image:hover .mwt-services-grid-image-wrap:after'  => 'top: {{SIZE}}{{UNIT}}; bottom:auto;',
				],
				'condition' => [
					'services_item_style' => 'second_layout',
					'show_service_counter' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'services_icons_style',
			[
				'label' => __('Icon', 'mwt-addons-for-elementor'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'services_icons_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em'],
				'range'       => [
					'px' => [
						'min' => 1,
						'max' => 120,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'label_block' => true,
				'selectors'   => [
					'{{WRAPPER}} .mwt-services-grid-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-grid-icon svg' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs('services_icons_style_tabs');

		$this->start_controls_tab(
			'services_icons_style_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_icons_style_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_icons_style_color_bg',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-icon>a' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-icon>div' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'services_icons_style_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_icons_style_color_hover',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-icon .mwt-services-grid-icon-link:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-grid-icon .mwt-services-grid-icon-link:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_icons_style_color_bg_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-icon a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_icons_style_color_border_hover',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-icon a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'services_icons_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-icon>a, .mwt-services-grid-icon>div',
			]
		);

		$this->add_control(
			'services_icons_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-icon>a, .mwt-services-grid-icon>div'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_icons_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-icon>a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-grid-icon>div' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_icons_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-icon'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'services_item_style!' => 'third_layout',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-services-grid-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-services-grid-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-services-grid-title, {{WRAPPER}} .mwt-services-grid-title > a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .mwt-services-grid-title',
			]
		);

		$this->add_responsive_control(
			'title_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'description_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-services-grid-description',
			]
		);

		$this->add_responsive_control(
			'description_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-description'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'services_button_style',
			[
				'label'     => __('Button', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'services_button_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-services-grid-button',
			]
		);

		$this->start_controls_tabs(
			'services_button_style_tabs'
		);

		$this->start_controls_tab(
			'services_button_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_button_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_button_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'services_button_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_button_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_button_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_button_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'services_button_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-button',
			]
		);

		$this->add_control(
			'services_button_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-button'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_button_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_button_min_width',
			[
				'label'      => __('Min Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'separator' => 'before',
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-button'  => 'min-width: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'services_filter_style',
			[
				'label'     => __('Filter', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'services_filter' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'services_filter_typo',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .filters a.category, .mwt-services-carousel-container .filters a.category',
			]
		);

		$this->start_controls_tabs('services_filters');

		$this->start_controls_tab(
			'services_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.category' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.category' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'services_filter_border',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .filters a.category, .mwt-services-carousel-container .filters a.category',
			]
		);

		$this->add_control(
			'services_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category, .mwt-services-carousel-container .filters a.category'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'services_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a:hover' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'services_filter_border_hover',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .filters a.category:hover, .mwt-services-carousel-container .filters a.category:hover',
			]
		);

		$this->add_control(
			'services_filter_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.category:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'services_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'services_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.active' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.active' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'services_filter_border_active',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .filters a.active, .mwt-services-carousel-container .filters a.active',
			]
		);

		$this->add_control(
			'services_filter_border_radius_active',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'services_filter_shadow',
				'selector' => '{{WRAPPER}} .mwt-services-grid-container .filters a.category, .mwt-services-carousel-container .filters a.category',
			]
		);

		$this->add_responsive_control(
			'services_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters a.category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .filters a.category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'services_filters_bottom_spacing',
			[
				'label'      => __('Filters Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-grid-container .filters'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-services-carousel-container .filters'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_prev_next_section',
			[
				'label' => esc_html__('Prev/Next', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'services_layout' => 'carousel',
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'prev_next_icons_position',
			[
				'label' => esc_html__('Prev/Next Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom',
				'options' => [
					'bottom' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'middle' => esc_html__('Middle', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'prev_next_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__('Start', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-start-h',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-center-h',
					],
					'end' => [
						'title' => esc_html__('End', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-end-h',
					],
					'space-between' => [
						'title' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-justify-space-between-h',
					],
				],
				'default' => 'start',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav' => 'justify-content: {{VALUE}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'bottom',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_btn_gap',
			[
				'label'      => __('Buttons Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav' => 'gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'bottom',
					'prev_next_align' => ['start', 'center', 'end'],
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_offset',
			[
				'label'      => __('Offset X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav' => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}}; width: calc(100% + {{SIZE}}{{UNIT}} * 2);',
				],
				'condition' => [
					'prev_next_icons_position' => 'bottom',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_offset_miffle',
			[
				'label'      => __('Offset X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav.position-middle button.owl-prev' => 'left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav.position-middle button.owl-next' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'middle',
				]
			]
		);

		$this->add_control(
			'hr',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'prev_next_typography',
				'selector' => '{{WRAPPER}} .mwt-services-carousel-container .owl-nav button',
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 4,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button svg' => 'font-size: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_control(
			'prev_next_icon_position',
			[
				'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'row' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'row-reverse' => esc_html__('Reverse', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button>span' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_spacing',
			[
				'label'      => __('Icon Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 12,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button>span' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'prev_next_style_tabs'
		);

		$this->start_controls_tab(
			'prev_next_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_disabled_tab',
			[
				'label' => esc_html__('Disabled', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_disabled',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_disabled',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_disabled',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_disabled',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button.disabled' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'prev_next_border',
				'selector' => '{{WRAPPER}} .mwt-services-carousel-container .owl-nav button',
			]
		);

		$this->add_control(
			'prev_next_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-nav' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'services_layout' => 'carousel',
					'data_nav_dots' => 'yes',
				]
			]
		);

		$this->start_controls_tabs(
			'dots_style_tabs'
		);

		$this->start_controls_tab(
			'dots_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dots_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dats_style_active_tab',
			[
				'label' => esc_html__('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-services-carousel-container .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Filter Cats
	 *
	 * Formats Category to be inserted in class attribute.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $string category slug.
	 *
	 * @return string $cat_filtered slug filtered.
	 */
	public function filter_cats($string)
	{

		$cat_filtered = trim($string);

		$cat_filtered = mb_strtolower($cat_filtered);

		if (strpos($cat_filtered, 'class') || strpos($cat_filtered, 'src')) {
			$cat_filtered = substr($cat_filtered, strpos($cat_filtered, '"') + 1);
			$cat_filtered = strtok($cat_filtered, '"');
			$cat_filtered = preg_replace('/[http:.]/', '', $cat_filtered);
			$cat_filtered = str_replace('/', '', $cat_filtered);
		}

		$cat_filtered = str_replace(', ', ',', $cat_filtered);
		$cat_filtered = preg_replace('/[\s_&@!#%]/', '-', $cat_filtered);
		$cat_filtered = str_replace(',', ' ', $cat_filtered);

		return $cat_filtered;
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$unique_id = uniqid();

		if (empty($settings['services-list'])) {
			return;
		}
		//carousel layout
		if ($settings['services_layout'] === 'carousel') : ?>
			<div class="mwt-services-carousel-container">
			<?php endif; //end carousel layout 

		//grid layout
		if ($settings['services_layout'] === 'grid') : ?>
				<div class="mwt-services-grid-container">
				<?php endif; //end grid layout 

			$filter = $settings['services_filter']; ?>
				<?php
				if ('yes' === $filter) :
				?>
					<div class="filters grid_filters-<?php echo esc_attr($unique_id); ?>">
						<a href="#" class="category selected active" data-filter="*">
							<?php echo wp_kses_post($settings['services_first_cat_label']); ?>
						</a>
						<?php
						foreach ($settings['services_cats_content'] as $index => $category) {
							if (!empty($category['services_cat'])) {
								$cat_filtered = $this->filter_cats($category['services_cat']);

								//start carousel layout 
								if ($settings['services_layout'] === 'carousel') : ?>
									<a href="#" class="category" data-filter=".<?php echo esc_attr($cat_filtered) ?>">
										<?php echo wp_kses_post($category['services_cat']); ?>
									</a>
								<?php endif;
								//end carousel layout 

								//start grid layout 
								if ($settings['services_layout'] === 'grid') : ?>
									<a href="#" class="category" data-filter="<?php echo esc_attr($cat_filtered) ?>">
										<?php echo wp_kses_post($category['services_cat']); ?>
									</a>
								<?php endif;
								//end grid layout 
								?>
						<?php
							}
						}
						?>
					</div>
				<?php
				endif;

				$show_service_counter = ($settings['show_service_counter'] === 'yes') ? 'with-service-counter' : '';

				//grid layout
				if ($settings['services_layout'] === 'grid') :
					$horizontal_order = ($settings['horizontal_order'] === 'yes') ? true : false;
				?>
					<div class="layout-cols layout-cols-<?php echo esc_attr($settings['items_per_row']); ?> layout-gap-<?php echo esc_attr($settings['layout_gap']) ?> <?php echo esc_attr($show_service_counter); ?>">
						<div class="grid-columns-wrapper">
							<div class="masonry" data-filters=".grid_filters-<?php echo esc_attr($unique_id); ?>" data-order="<?php echo esc_attr($horizontal_order); ?>">
								<div class="grid-sizer"></div>
							<?php endif;
						//end grid layout 

						//carousel layout
						if ($settings['services_layout'] === 'carousel') :
							$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
							$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
							$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
							$data_nav = ($settings['data_nav_prev_next'] === 'yes') ? 'true' : 'false';
							$data_dots = ($settings['data_nav_dots'] === 'yes') ? 'true' : 'false';
							$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
							$data_rtl = ($settings['data_rtl'] === 'yes') ? 'true' : 'false'; ?>
								<div class="owl-carousel mwt-services-carousel <?php echo esc_attr($show_service_counter); ?>" data-number="<?php echo esc_attr($unique_id); ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-md="<?php echo esc_attr($settings['data_responsive_md']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".grid_filters-<?php echo esc_attr($unique_id); ?>">
									<?php endif;
								//end carousel layout  

								foreach ($settings['services-list'] as $index => $item) {
									//carousel layout
									if ($settings['services_layout'] === 'carousel') : ?>
										<div class="owl-carousel-item <?php echo esc_attr($settings['services_item_style']) ?> <?php echo esc_attr($this->filter_cats($item['services_item_category'])) ?>">
										<?php endif;
									//end carousel layout 

									//grid layout
									if ($settings['services_layout'] === 'grid') : ?>
											<div class="grid-item <?php echo esc_attr($settings['services_item_style']) ?> <?php echo esc_attr($this->filter_cats($item['services_item_category'])) ?>">
												<?php endif;
											//end grid layout  

											$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['services_bg_image']['id'], 'thumbnail', $settings);
											if (!$image_url && isset($item['services_bg_image']['url'])) {
												$image_url = $item['services_bg_image']['url'];
											}
											$bg_image_html = 'style="background-image: url(' . esc_url($image_url) . ')"';

											if (!empty($item['item_name'])) {
												$image_alt = $item['services_item_title'];
											} else {
												$image_alt = 'This is a name';
											}
											$image_html = '<img class="mwt-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';

											$show_corners = ($settings['show_corners'] === 'yes') ? 'with-corners' : '';
											$show_top_border_hover = ($settings['show_top_border_hover'] === 'yes') ? 'with-top-border-hover' : '';

											$services_item_style = $settings['services_item_style'];
											switch ($services_item_style) {
												case 'default': ?>
													<div class="mwt-services-grid-item-wrap <?php echo esc_attr($show_top_border_hover) ?>" <?php echo (!empty($image_url)) ? $bg_image_html : '' ?>>
														<div class="mwt-services-grid-content-wrap <?php echo esc_attr($show_corners); ?>">
															<?php
															$services_item_icon = isset($settings['services-list'][$index]['services_item_icon']) ? $settings['services-list'][$index]['services_item_icon'] : '';
															$title = $settings['services-list'][$index]['services_item_title'];
															$title_link = $settings['services-list'][$index]['services_item_title_link']['url'];
															$description = $settings['services-list'][$index]['services_item_description'];
															$button = $settings['services-list'][$index]['services_item_button'];
															$button_link = !empty($settings['services-list'][$index]['services_item_button_link']['url']) ? $settings['services-list'][$index]['services_item_button_link']['url'] : '#';

															if ($services_item_icon) : ?>
																<div class="mwt-services-grid-icon">
																	<?php if (!empty($title_link)) :
																		echo '<a href="' . esc_attr($title_link) . '" class="mwt-services-grid-icon-link">';
																	else :
																		echo '<div>';
																	endif;
																	\Elementor\Icons_Manager::render_icon($settings['services-list'][$index]['services_item_icon'], ['aria-hidden' => 'true', 'class' => 'services-item-icon']);
																	if (!empty($title_link)) :
																		echo '</a>';
																	else :
																		echo '</div>';
																	endif; ?>
																</div>
															<?php endif;
															if (!empty($title_link)) { ?>
																<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-title">
																	<?php echo esc_html($title); ?>
																</a>
															<?php } else { ?>
																<div class="mwt-services-grid-title">
																	<?php echo esc_html($title); ?>
																</div>
															<?php };
															if ($settings['show_divider'] === 'yes') : ?>
																<div class="mwt-services-grid-divider"><span></span></div>
															<?php endif;
															if (!empty($description)) : ?>
																<div class="mwt-services-grid-description">
																	<?php echo esc_html($description); ?>
																</div>
															<?php endif;
															if (!empty($button)) : ?>
																<a href="<?php echo esc_attr($button_link) ?>" class="mwt-services-grid-button">
																	<?php echo esc_html($button); ?>
																</a>
															<?php endif; ?>
														</div>
													</div>
												<?php break;

												case 'second_layout':

													$services_item_icon = !empty($settings['services-list'][$index]['services_item_icon']['value']) ? $settings['services-list'][$index]['services_item_icon'] : '';
													$title = $settings['services-list'][$index]['services_item_title'];
													$title_link = $settings['services-list'][$index]['services_item_title_link']['url'];
													$description = $settings['services-list'][$index]['services_item_description'];
													$button = $settings['services-list'][$index]['services_item_button'];
													$button_link = !empty($settings['services-list'][$index]['services_item_button_link']['url']) ? $settings['services-list'][$index]['services_item_button_link']['url'] : '#';
													$services_image_entire_item = ($settings['services-list'][$index]['services_image_entire_item'] === 'yes') ? 'item-entire-image' : ''; ?>

													<div class="mwt-services-grid-item-wrap <?php echo esc_attr($show_top_border_hover . ' ' . $services_image_entire_item) ?>">
														<?php if (!empty($image_url)) { ?>
															<div class="mwt-services-grid-image-wrap image-absolute">
																<?php echo wp_kses_post($image_html); ?>
																<?php if (!empty($title_link)) : ?>
																	<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-image-link"></a>
																<?php endif; ?>
															</div>
															<div class="mwt-services-grid-image-wrap">
																<?php echo wp_kses_post($image_html); ?>
																<?php if (!empty($title_link)) : ?>
																	<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-image-link"></a>
																<?php endif; ?>
															</div>
														<?php } ?>
														<?php if (!empty($services_item_icon) || !empty($title) || !empty($description) || !empty($button)) { ?>
															<div class="mwt-services-grid-content-wrap <?php echo esc_attr($show_corners); ?>">
																<?php
																if ($services_item_icon) : ?>
																	<div class="mwt-services-grid-icon">
																		<?php if (!empty($title_link)) :
																			echo '<a href="' . esc_attr($title_link) . '" class="mwt-services-grid-icon-link">';
																		else :
																			echo '<div>';
																		endif;
																		\Elementor\Icons_Manager::render_icon($settings['services-list'][$index]['services_item_icon'], ['aria-hidden' => 'true', 'class' => 'services-item-icon']);
																		if (!empty($title_link)) :
																			echo '</a>';
																		else :
																			echo '</div>';
																		endif; ?>
																	</div>
																<?php endif;
																if (!empty($title_link)) { ?>
																	<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-title">
																		<?php echo esc_html($title); ?>
																	</a>
																<?php } else { ?>
																	<div class="mwt-services-grid-title">
																		<?php echo esc_html($title); ?>
																	</div>
																<?php };
																if ($settings['show_divider'] === 'yes') : ?>
																	<div class="mwt-services-grid-divider"><span></span></div>
																<?php endif;
																if (!empty($description)) : ?>
																	<div class="mwt-services-grid-description">
																		<?php echo esc_html($description); ?>
																	</div>
																<?php endif;
																if (!empty($button)) : ?>
																	<a href="<?php echo esc_attr($button_link) ?>" class="mwt-services-grid-button">
																		<?php echo esc_html($button); ?>
																	</a>
																<?php endif; ?>
															</div>
														<?php } ?>
													</div>
												<?php break;

												case 'third_layout':

													$services_item_icon = !empty($settings['services-list'][$index]['services_item_icon']['value']) ? $settings['services-list'][$index]['services_item_icon'] : '';
													$title = $settings['services-list'][$index]['services_item_title'];
													$title_link = $settings['services-list'][$index]['services_item_title_link']['url'];
													$description = $settings['services-list'][$index]['services_item_description'];
													$button = $settings['services-list'][$index]['services_item_button'];
													$button_link = !empty($settings['services-list'][$index]['services_item_button_link']['url']) ? $settings['services-list'][$index]['services_item_button_link']['url'] : '#';
													$services_image_entire_item = ($settings['services-list'][$index]['services_image_entire_item'] === 'yes') ? 'item-entire-image' : ''; ?>

													<div class="mwt-services-grid-item-wrap <?php echo esc_attr($show_top_border_hover . ' ' . $services_image_entire_item) ?>">
														<?php if (!empty($image_url)) { ?>
															<div class="mwt-services-grid-image-wrap image-absolute">
																<?php echo wp_kses_post($image_html); ?>
																<?php if (!empty($title_link)) : ?>
																	<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-image-link"></a>
																<?php endif; ?>
															</div>
															<div class="mwt-services-grid-image-wrap">
																<?php echo wp_kses_post($image_html); ?>
																<?php if (!empty($title_link)) : ?>
																	<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-image-link"></a>
																<?php endif; ?>
															</div>
														<?php } ?>
														<?php if (!empty($services_item_icon) || !empty($title) || !empty($description) || !empty($button)) { ?>
															<div class="mwt-services-grid-content-wrap <?php echo esc_attr($show_corners); ?>">
																<?php
																if ($services_item_icon) : ?>
																	<div class="mwt-services-grid-icon">
																		<?php if (!empty($title_link)) :
																			echo '<a href="' . esc_attr($title_link) . '" class="mwt-services-grid-icon-link">';
																		else :
																			echo '<div>';
																		endif;
																		\Elementor\Icons_Manager::render_icon($settings['services-list'][$index]['services_item_icon'], ['aria-hidden' => 'true', 'class' => 'services-item-icon']);
																		if (!empty($title_link)) :
																			echo '</a>';
																		else :
																			echo '</div>';
																		endif; ?>
																	</div>
																<?php endif; ?>
																<div class="mwt-services-right-column">
																	<?php
																	if (!empty($title_link)) { ?>
																		<a href="<?php echo esc_attr($title_link) ?>" class="mwt-services-grid-title">
																			<?php echo esc_html($title); ?>
																		</a>
																	<?php } else { ?>
																		<div class="mwt-services-grid-title">
																			<?php echo esc_html($title); ?>
																		</div>
																	<?php };
																	if ($settings['show_divider'] === 'yes') : ?>
																		<div class="mwt-services-grid-divider"><span></span></div>
																	<?php endif;
																	if (!empty($description)) : ?>
																		<div class="mwt-services-grid-description">
																			<?php echo esc_html($description); ?>
																		</div>
																	<?php endif;
																	if (!empty($button)) : ?>
																		<a href="<?php echo esc_attr($button_link) ?>" class="mwt-services-grid-button">
																			<?php echo esc_html($button); ?>
																		</a>
																	<?php endif; ?>
																</div>
															</div>
														<?php } ?>
													</div>
											<?php break;
											} ?>
											</div>
										<?php } ?>
										</div>
										<?php
										//carousel layout 
										if ($settings['services_layout'] === 'carousel') : ?>
											<div class="owl-nav position-<?php echo esc_attr($settings['prev_next_icons_position']); ?> mwt-services-carousel-nav-<?php echo esc_attr($unique_id); ?>">
												<?php
												$date_nav_prev = isset($settings['date_nav_prev']) ? $settings['date_nav_prev'] : '';
												$date_nav_next = isset($settings['date_nav_next']) ? $settings['date_nav_next'] : '';
												$date_nav_prev_word = isset($settings['date_nav_prev_word']) ? $settings['date_nav_prev_word'] : '';
												$date_nav_next_word = isset($settings['date_nav_next_word']) ? $settings['date_nav_next_word'] : '';
												?>
												<span class="mwt-services-carousel-prev-<?php echo esc_attr($unique_id); ?>">
													<?php if ($date_nav_prev) :
														\Elementor\Icons_Manager::render_icon($settings['date_nav_prev'], ['aria-hidden' => 'true', 'class' => 'icon-prev']);
													endif;
													echo esc_html($date_nav_prev_word)
													?>
												</span>
												<span class="mwt-services-carousel-next-<?php echo esc_attr($unique_id); ?>">
													<?php
													echo esc_html($date_nav_next_word);
													if ($date_nav_next) :
														\Elementor\Icons_Manager::render_icon($settings['date_nav_next'], ['aria-hidden' => 'true', 'class' => 'icon-next']);
													endif;
													?>
												</span>
											</div>
										<?php endif; //end carousel layout 
										?>
										<?php
										//grid layout 
										if ($settings['services_layout'] === 'grid') : ?>
								</div>
							</div>
						<?php endif; //end grid layout 
						?>
						</div>
				<?php }
		}
