<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Products Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Products_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['mwt-products-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['mwt-products-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-products-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Products Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slider-push';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'visual', 'products', 'carousel', 'slider', 'woocommerce'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'general_settings',
			[
				'label' => __('General', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'layout_type',
			[
				'label' => esc_html__('Products Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'label_block' => true,
				'default' => 'default',
				'options' => [
					'default' => esc_html__('Carousel', 'mwt-addons-for-elementor'),
					'cat-layout' => esc_html__('Carousel of categories', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'type_products',
			[
				'label' => esc_html__('Type', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'on_sale' => esc_html__('On Sale', 'mwt-addons-for-elementor'),
					'best_selling'  => esc_html__('Best Selling', 'mwt-addons-for-elementor'),
					'top_rated' => esc_html__('Top Rated', 'mwt-addons-for-elementor'),
					'featured_products' => esc_html__('Featured', 'mwt-addons-for-elementor'),
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'number_of_products',
			[
				'label'       => __('Limit', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'min'         => 1,
				'default'     => 6,
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'products_category',
			[
				'label' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'description' => __('To assign for multiple categories, separate by a comma \',\'', 'mwt-addons-for-elementor'),
				'placeholder' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'default' => '',
				'label_block' => true,
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_rtl',
			[
				'label'   => __('Enable RTL', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'frontend_available' => true,
				'condition' => [
					'data_autoplay' => 'yes',
				]
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 4,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_md',
			[
				'label' => esc_html__('Laptop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tablet Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 80,
				'step' => 1,
				'default' => 30,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'orderby_products',
			[
				'label' => esc_html__('Order by', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'date',
				'separator' => 'before',
				'options' => [
					'title'      => esc_html__('Title', 'mwt-addons-for-elementor'),
					'date'       => esc_html__('Date', 'mwt-addons-for-elementor'),
					'id'         => esc_html__('Id', 'mwt-addons-for-elementor'),
					'menu_order' => esc_html__('Menu Order', 'mwt-addons-for-elementor'),
					'popularity' => esc_html__('Popularity', 'mwt-addons-for-elementor'),
					'rand'       => esc_html__('Randomly', 'mwt-addons-for-elementor'),
					'rating'     => esc_html__('Rating', 'mwt-addons-for-elementor'),
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'order_products',
			[
				'label' => esc_html__('Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'ASC',
				'options' => [
					'ASC'  => esc_html__('ASC', 'mwt-addons-for-elementor'),
					'DESC' => esc_html__('DESC', 'mwt-addons-for-elementor'),
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'data_nav_prev_next',
			[
				'label'   => __('Show prev/next', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'date_nav_prev',
			[
				'label'            => __('Previous Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_prev_word',
			[
				'label'     => __('Previous word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Prev', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'date_nav_next',
			[
				'label'            => __('Next Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_next_word',
			[
				'label'     => __('Next word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Next', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'data_nav_dots',
			[
				'label'   => __('Show dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_background_color',
			[
				'label' => esc_html__('Item Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container ul.products li.product .product-loop-item' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_image_background_color',
			[
				'label' => esc_html__('Image Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .product-thumbnail-wrap>a::before' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'product_title_typography',
				'selector' =>
				'{{WRAPPER}} .woocommerce-loop-product__title',
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-loop-product__title a' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-loop-product__title a:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_responsive_control(
			'items_title_min_height',
			[
				'label'      => __('Product Title Min Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 1,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .woocommerce-loop-product__title' => 'min-height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_description',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'show_product_description',
			[
				'label' => esc_html__('Show Description', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'block',
				'options' => [
					'block' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'none' => esc_html__('None', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .product-short-description' => 'display: {{VALUE}} !important;',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'product_description_typography',
				'selector' =>
				'{{WRAPPER}} .product-short-description',
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-short-description' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_description_line_clamp',
			[
				'label' => esc_html__('Line Limit', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'on' => esc_html__('On', 'mwt-addons-for-elementor'),
				'off' => esc_html__('Off', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'product_description_line_clamp_select',
			[
				'label' => esc_html__('Number of Lines', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => '2',
				'options' => [
					'1' => esc_html__('1', 'mwt-addons-for-elementor'),
					'2'  => esc_html__('2', 'mwt-addons-for-elementor'),
					'3' => esc_html__('3', 'mwt-addons-for-elementor'),
					'4' => esc_html__('4', 'mwt-addons-for-elementor'),
					'5' => esc_html__('5', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .product-short-description' => '
					overflow: hidden; 
					display: -webkit-box !important;
					-webkit-line-clamp: {{VALUE}};
							line-clamp: {{VALUE}}; 
					-webkit-box-orient: vertical;',
				],
				'condition' => [
					'layout_type!' => 'cat-layout',
					'product_description_line_clamp' => 'yes',
				],
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'show_product_button',
			[
				'label' => esc_html__('Show Button', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'block',
				'options' => [
					'block' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'none' => esc_html__('None', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .shop-product-buttons' => 'display: {{VALUE}} !important;',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'show_product_button',
			[
				'label' => esc_html__('Show Button', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'block',
				'options' => [
					'block' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'none' => esc_html__('None', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .shop-product-buttons' => 'display: {{VALUE}} !important;',
				],
				'condition' => [
					'layout_type!' => 'cat-layout'
				]
			]
		);

		$this->add_responsive_control(
			'item_content_align',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__('Left', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-h-align-center',
					],
					'right' => [
						'title' => esc_html__('Right', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);


		$this->add_control(
			'item_overlay',
			[
				'label' => esc_html__('Overlay', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item .item-media::before' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_bg',
			[
				'label' => esc_html__('Content Background', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_responsive_control(
			'item_content_padding',
			[
				'label'      => __('Content Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_responsive_control(
			'item_content_margin',
			[
				'label'      => __('Content Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_title_options',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'item_content_title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap h6, .mwt-products-carousel-categories-item-title-wrap h6 a',
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap h6' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap h6 a' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap h6 a:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_text_options',
			[
				'label' => esc_html__('Text', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'item_content_text_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap p',
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->add_control(
			'item_content_text_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-categories-item-title-wrap p' => 'color: {{VALUE}};',
				],
				'condition' => [
					'layout_type' => 'cat-layout'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_prev_next_section',
			[
				'label' => esc_html__('Prev/Next', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'prev_next_icons_position',
			[
				'label' => esc_html__('Prev/Next Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom',
				'options' => [
					'bottom' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'middle' => esc_html__('Middle', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'prev_next_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__('Start', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-start-h',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-center-h',
					],
					'end' => [
						'title' => esc_html__('End', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-end-h',
					],
					'space-between' => [
						'title' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-justify-space-between-h',
					],
				],
				'default' => 'start',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav' => 'justify-content: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_offset',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav' => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}}; width: calc(100% + {{SIZE}}{{UNIT}} * 2);',
				],
			]
		);

		$this->add_control(
			'hr',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'prev_next_typography',
				'selector' => '{{WRAPPER}} .mwt-products-carousel-container .owl-nav button',
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 4,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button svg' => 'font-size: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_control(
			'prev_next_icon_position',
			[
				'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'row' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'row-reverse' => esc_html__('Reverse', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button>span' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_spacing',
			[
				'label'      => __('Icon Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 12,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button>span' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'prev_next_style_tabs'
		);

		$this->start_controls_tab(
			'prev_next_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_disabled_tab',
			[
				'label' => esc_html__('Disabled', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_disabled',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_disabled',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_disabled',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_disabled',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button.disabled' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'prev_next_border',
				'selector' => '{{WRAPPER}} .mwt-products-carousel-container .owl-nav button',
			]
		);

		$this->add_control(
			'prev_next_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_btn_gap',
			[
				'label'      => __('Buttons Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-nav' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_dots' => 'yes',
				]
			]
		);

		$this->start_controls_tabs(
			'dots_style_tabs'
		);

		$this->start_controls_tab(
			'dots_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dots_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dats_style_active_tab',
			[
				'label' => esc_html__('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-products-carousel-container .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$unique_id = uniqid();

		$products_params = '';

		$layout_type = $settings['layout_type'];

		if ($settings['type_products'] == 'on_sale') {
			$products_params .= ' on_sale="true"';
		} elseif ($settings['type_products'] == 'best_selling') {
			$products_params .= ' best_selling="true"';
		} elseif ($settings['type_products'] == 'top_rated') {
			$products_params .= ' top_rated="true"';
		} elseif ($settings['type_products'] == 'featured_products') {
			$products_params .= ' visibility="featured"';
		}

		$limit = $settings['number_of_products'];
		if ($limit) {
			$products_params .= ' limit="' . $limit . '"';
		}

		$products_params .= ' columns="1"';
		$products_params .= ' orderby="' . $settings['orderby_products'] . '"';
		$products_params .= ' order="' . $settings['order_products'] . '"';
		$products_params .= ' category="' . $settings['products_category'] . '"';

		$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
		$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
		$data_rtl = ($settings['data_rtl'] === 'yes') ? 'true' : 'false';
		$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
		$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
		$data_nav_prev_next = ($settings['data_nav_prev_next'] === 'yes') ? 'true' : 'false';
		$data_nav_dots = ($settings['data_nav_dots'] === 'yes') ? 'true' : 'false';
?>
		<div class="mwt-products-carousel-container">
			<div class="mwt-products-carousel" data-number="<?php echo esc_attr($unique_id); ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav_prev_next) ?>" data-dots="<?php echo esc_attr($data_nav_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-md="<?php echo esc_attr($settings['data_responsive_md']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">

				<?php switch ($layout_type):
					case 'default':
						echo do_shortcode('[products' . $products_params . ']');
						break;

					case 'cat-layout': ?>
						<div class="woocommerce">
							<ul class="products">
								<?php
								$taxonomy_name = "product_cat";
								$product_categories = get_terms($taxonomy_name, array(
									'orderby' => 'name',
									'order' => 'ASC',
									'hide_empty' => 1
								));
								foreach ($product_categories as $cat) {
									$var = get_term($cat);
									$cat_thumb_id = get_term_meta($cat->term_id, 'thumbnail_id', true);
									$count = $var->count;
								?>
									<li class="mwt-products-carousel-categories-item">
										<div class="position-relative">
											<div class="item-media">
												<?php
												if (!empty(wp_get_attachment_image($cat_thumb_id, array('600', '600')))) :
													echo wp_get_attachment_image($cat_thumb_id, array('600', '600'));
												else :
												?>
													<img src="<?php echo esc_attr(MWT_ADDONS_PLUGIN_URL . 'assets/img/woocommerce-placeholder.png') ?>" alt="<?php echo esc_attr('placeholder') ?>">
												<?php endif; ?>
											</div>
											<div class="mwt-products-carousel-categories-item-title-wrap">
												<h6>
													<a href="<?php echo esc_attr(get_category_link($cat)) ?>"><?php echo esc_attr($var->name) ?></a>
												</h6>
												<p>
													<?php
													$letter = $count . ' ' . (($count > 1) ? 'products' : 'product');
													echo esc_html($letter)
													?>
												</p>
											</div>
										</div>
									</li>
								<?php
								} //foreach
								?>
							</ul>
						</div>
				<?php
						break;
				endswitch;
				?>
				<div class="owl-nav position-<?php echo esc_attr($settings['prev_next_icons_position']) ?>" id="mwt-products-carousel-nav-<?php echo esc_attr($unique_id); ?>">
					<?php
					$date_nav_prev = isset($settings['date_nav_prev']) ? $settings['date_nav_prev'] : '';
					$date_nav_next = isset($settings['date_nav_next']) ? $settings['date_nav_next'] : '';
					$date_nav_prev_word = isset($settings['date_nav_prev_word']) ? $settings['date_nav_prev_word'] : '';
					$date_nav_next_word = isset($settings['date_nav_next_word']) ? $settings['date_nav_next_word'] : '';
					?>
					<span class="mwt-products-carousel-prev-<?php echo esc_attr($unique_id); ?>">
						<?php if ($date_nav_prev) :
							\Elementor\Icons_Manager::render_icon($settings['date_nav_prev'], ['aria-hidden' => 'true', 'class' => 'icon-prev']);
						endif;
						echo esc_html($date_nav_prev_word)
						?>
					</span>
					<span class="mwt-products-carousel-next-<?php echo esc_attr($unique_id); ?>">
						<?php
						echo esc_html($date_nav_next_word);
						if ($date_nav_next) :
							\Elementor\Icons_Manager::render_icon($settings['date_nav_next'], ['aria-hidden' => 'true', 'class' => 'icon-next']);
						endif;
						?>
					</span>
				</div>

			</div>
			<?php wp_reset_postdata(); ?>
		</div>
<?php }
}
