<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Posts Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Posts_Accordion extends \Elementor\Widget_Base

{
    /**
     * Get widget name.
     *
     * Retrieve list widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'mwt-posts-accordion';
    }

    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['mwt-posts-accordion-style'];
    }

    /**
     * Get widget title.
     *
     * Retrieve list widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Posts Accordion', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve image post widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-posts-group';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'visual', 'posts', 'carousel', 'grid', 'slider'];
    }

    /**
     * Register list widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        $this->start_controls_section(
            'general_settings',
            [
                'label' => __('General', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'number_of_posts',
            [
                'label'       => __('Posts Per Page', 'mwt-addons-for-elementor'),
                'description' => __('Set the number of per page', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::NUMBER,
                'min'         => 2,
                'max'         => 5,
                'default'     => 5,
            ]
        );

        $this->add_control(
            'posts_layout_gap',
            [
                'label' => esc_html__('Gap(px)', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::NUMBER,
                'min'         => 0,
                'max'         => 80,
                'default'     => 30,
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-container' => 'gap: {{VALUE}}px;',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'exclude' => ['custom'],
                'include' => [],
                'default' => 'full',
            ]
        );

        $this->add_control(
            'show_date',
            [
                'label'     => __('Show Date', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'show_author',
            [
                'label'     => __('Show Author', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'author_icon',
            [
                'label'            => __('Author Icon', 'mwt-addons-for-elementor'),
                'type'             => \Elementor\Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'fas fa-user',
                    'library' => 'fa-solid',
                ],
                'recommended'      => [
                    'fa-solid'   => [
                        'user',
                    ],
                ],
                'skin'             => 'inline',
                'label_block'      => false,
                'condition' => [
                    'show_author' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'show_comments',
            [
                'label'     => __('Show Comments', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'comments_icon',
            [
                'label'            => __('Comments Icon', 'mwt-addons-for-elementor'),
                'type'             => \Elementor\Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'fas fa-comment',
                    'library' => 'fa-solid',
                ],
                'recommended'      => [
                    'fa-solid'   => [
                        'comment',
                    ],
                ],
                'skin'             => 'inline',
                'label_block'      => false,
                'condition' => [
                    'show_comments' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'show_likes',
            [
                'label'     => __('Show Likes', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'likes_icon',
            [
                'label'            => __('Likes Icon', 'mwt-addons-for-elementor'),
                'type'             => \Elementor\Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'far fa-heart',
                    'library' => 'fa-regular',
                ],
                'recommended'      => [
                    'fa-regular'   => [
                        'heart',
                    ],
                ],
                'skin'             => 'inline',
                'label_block'      => false,
                'condition' => [
                    'show_likes' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'show_views',
            [
                'label'     => __('Show Views', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'views_icon',
            [
                'label'            => __('Views Icon', 'mwt-addons-for-elementor'),
                'type'             => \Elementor\Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'far fa-eye',
                    'library' => 'fa-regular',
                ],
                'recommended'      => [
                    'fa-regular'   => [
                        'eye',
                    ],
                ],
                'skin'             => 'inline',
                'label_block'      => false,
                'condition' => [
                    'show_views' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'posts_excerpt_length',
            [
                'label'     => __('Excerpt Length', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::NUMBER,
                'default'   => 30,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'animation_section',
            [
                'label' => __('Entrance Animation', 'mwt-addons-for-elementor'),
            ]
        );

        $entrance_animation = [
            ''             => esc_html__('None', 'mwt-addons-for-elementor'),
            'bounce'       => esc_html__('bounce', 'mwt-addons-for-elementor'),
            'flash'        => esc_html__('flash', 'mwt-addons-for-elementor'),
            'pulse'        => esc_html__('pulse', 'mwt-addons-for-elementor'),
            'rubberBand'   => esc_html__('rubberBand', 'mwt-addons-for-elementor'),
            'shake'        => esc_html__('shake', 'mwt-addons-for-elementor'),
            'headShake'    => esc_html__('headShake', 'mwt-addons-for-elementor'),
            'swing'        => esc_html__('swing', 'mwt-addons-for-elementor'),
            'tada'         => esc_html__('tada', 'mwt-addons-for-elementor'),
            'wobble'       => esc_html__('wobble', 'mwt-addons-for-elementor'),
            'jello'        => esc_html__('jello', 'mwt-addons-for-elementor'),
            'heartBeat'    => esc_html__('heartBeat', 'mwt-addons-for-elementor'),
            'bounceIn'     => esc_html__('bounceIn', 'mwt-addons-for-elementor'),
            'fadeIn'       => esc_html__('fadeIn', 'mwt-addons-for-elementor'),
            'fadeInDown'   => esc_html__('fadeInDown', 'mwt-addons-for-elementor'),
            'fadeInLeft'   => esc_html__('fadeInLeft', 'mwt-addons-for-elementor'),
            'fadeInRight'  => esc_html__('fadeInRight', 'mwt-addons-for-elementor'),
            'fadeInUp'     => esc_html__('fadeInUp', 'mwt-addons-for-elementor'),
            'flip'         => esc_html__('flip', 'mwt-addons-for-elementor'),
            'flipInX'      => esc_html__('flipInX', 'mwt-addons-for-elementor'),
            'flipInY'      => esc_html__('flipInY', 'mwt-addons-for-elementor'),
            'lightSpeedIn' => esc_html__('lightSpeedIn', 'mwt-addons-for-elementor'),
            'jackInTheBox' => esc_html__('jackInTheBox', 'mwt-addons-for-elementor'),
            'zoomIn'       => esc_html__('zoomIn', 'mwt-addons-for-elementor'),
            'pullUp'       => esc_html__('pullUp', 'mwt-addons-for-elementor'),
        ];

        $animation_duration = [
            'slower'   => esc_html__('Slower', 'mwt-addons-for-elementor'),
            'slow'   => esc_html__('Slow', 'mwt-addons-for-elementor'),
            ''   => esc_html__('Normal', 'mwt-addons-for-elementor'),
            'fast'   => esc_html__('Fast', 'mwt-addons-for-elementor'),
            'faster'   => esc_html__('Faster', 'mwt-addons-for-elementor'),
        ];

        $this->add_control(
            'results_section_date_animation',
            [
                'label' => esc_html__('Date (Results Section)', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '',
                'options' => $entrance_animation,
                'frontend_available' => true,
            ]
        );

        $this->add_control(
            'results_section_date_animation_duration',
            [
                'label' => esc_html__('Animation Duration', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '',
                'options' => $animation_duration,
                'frontend_available' => true,
                'condition' => [
                    'results_section_date_animation!' => '',
                ]
            ]
        );

        $this->add_control(
            'results_section_content_animation',
            [
                'label' => esc_html__('Content (Results Section)', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '',
                'options' => $entrance_animation,
                'frontend_available' => true,
            ]
        );

        $this->add_control(
            'results_section_content_animation_duration',
            [
                'label' => esc_html__('Animation Duration', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '',
                'options' => $animation_duration,
                'frontend_available' => true,
                'condition' => [
                    'results_section_content_animation!' => '',
                ]
            ]
        );

        $this->add_control(
            'results_section_animation_delay',
            [
                'label' => esc_html__('Animation Delay', 'mwt-addons-for-elementor') . ' (ms)',
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 300,
                'step' => 100,
                'default' => 300,
                'separator' => 'before',
                'frontend_available' => true,
            ]
        );

        $this->end_controls_section();

        // style
        $this->start_controls_section(
            'style_general_section',
            [
                'label' => esc_html__('General', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_date' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'meta_transition',
            [
                'label'      => __('Transition Duration', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'size' => 0.3,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-container a'  => 'transition: all {{SIZE}}s',
                    '{{WRAPPER}} .mwt-posts-accordion-container a i'  => 'transition: all {{SIZE}}s',
                    '{{WRAPPER}} .mwt-posts-accordion-container a svg'  => 'transition: all {{SIZE}}s',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_date_section',
            [
                'label' => esc_html__('Date', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_date' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs(
            'date_style_tabs'
        );

        $this->start_controls_tab(
            'date_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'date_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-date' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-date' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'date_bg_color',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-date' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-date' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'date_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'date_color_hover',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-date:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-date:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'date_bg_color_hover',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-date:hover' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-date:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'date_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-date' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-date' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'date_day',
            [
                'label' => esc_html__('Day', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'date_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-image-item-date span:first-child, .mwt-posts-accordion-nav-item-date span:first-child',
            ]
        );

        $this->add_control(
            'date_month',
            [
                'label' => esc_html__('Month', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'month_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-image-item-date span:last-child, .mwt-posts-accordion-nav-item-date span:last-child',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_results_section',
            [
                'label' => esc_html__('Results Section', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'results_section_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'overlay_color',
            [
                'label' => esc_html__('Overlay', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item::before' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_options',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'title_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-image-item-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_bottom_spacing',
            [
                'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'meta_options',
            [
                'label' => esc_html__('Meta', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'meta_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-image-item-footer',
            ]
        );

        $this->add_responsive_control(
            'meta_icon_spacing',
            [
                'label'      => __('Icon Right Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 3,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer i'  => 'margin-right: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer svg'  => 'margin-right: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'meta_spacing',
            [
                'label'      => __('Spacing Between Meta Elements', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 12,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer-end-block span+span'  => 'margin-left: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'meta_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'meta_color_hover',
            [
                'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer a:hover i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-image-item-footer a:hover svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_navigation_section',
            [
                'label' => esc_html__('Navigation Section', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'navigation_section_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'nav_bg_color',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-top-wrap' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'nav_border',
                'selector' => '{{WRAPPER}} .mwt-posts-accordion-nav-item-top-wrap',
            ]
        );

        $this->add_control(
            'nav_border_radius',
            [
                'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-top-wrap'  => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'nav_bottom_spacing',
            [
                'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item+.mwt-posts-accordion-nav-item'  => 'margin-top: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'nav_title_options',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'nav_title_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-nav-item-title',
            ]
        );

        $this->add_control(
            'nav_title_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-title' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item.active:hover' => 'color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'nav_title_color_hover',
            [
                'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-title:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'nav_title_bottom_spacing',
            [
                'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'nav_meta_options',
            [
                'label' => esc_html__('Meta', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'nav_meta_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer',
            ]
        );

        $this->add_responsive_control(
            'nav_meta_icon_spacing',
            [
                'label'      => __('Icon Right Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 3,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer i'  => 'margin-right: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer svg'  => 'margin-right: {{SIZE}}{{UNIT}}',

                ],
            ]
        );

        $this->add_responsive_control(
            'nav_meta_spacing',
            [
                'label'      => __('Spacing Between Meta Elements', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 12,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer-end-block span+span'  => 'margin-left: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'nav_meta_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'nav_meta_color_hover',
            [
                'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer a:hover i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-footer a:hover svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'nav_excerpt_options',
            [
                'label' => esc_html__('Excerpt', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'nav_excerpt_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt',
            ]
        );

        $this->add_control(
            'nav_excerpt_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'nav_excerpt_bg_color',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'nav_excerpt_border',
                'selector' => '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt div',
            ]
        );

        $this->add_control(
            'nav_excerpt_border_radius',
            [
                'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt'  => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'nav_excerpt_padding',
            [
                'label'      => __('Excerpt Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item-excerpt div' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'nav_excerpt_active_max_height',
            [
                'label'      => __('Max Height When Active (px)', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 60,
                        'max' => 1000,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 200,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-posts-accordion-nav-item.active .mwt-posts-accordion-nav-item-excerpt'  => 'max-height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render list widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $unique_id = uniqid();

        $mwt_posts = new WP_Query(array(
            'post_type'           => 'post',
            'orderby'             => 'post_date',
            'order '              => 'DESC',
            'ignore_sticky_posts' => true,
            'posts_per_page'      => $settings['number_of_posts'],
            'tax_query' => array(
                array(
                    'taxonomy' => 'post_format',
                    'field' => 'slug',
                    'terms' => array('post-format-quote', 'post-format-status', 'post-format-link', 'post-format-aside', 'post-format-chat'),
                    'operator' => 'NOT IN'
                )
            )
        ));

        if (!$mwt_posts->have_posts()) :
            esc_html_e('No posts found', 'mwt-addons-for-elementor');
            return;
        endif; ?>
        <div class="mwt-posts-accordion-container">
            <div class="mwt-posts-accordion-images">
                <?php
                $post_count = 0;
                while ($mwt_posts->have_posts()) :
                    $mwt_posts->the_post();
                    //animation
                    $animation_delay = (!empty($settings['results_section_animation_delay'])) ? $settings['results_section_animation_delay'] : '';
                    $date_animation = (!empty($settings['results_section_date_animation'])) ? $settings['results_section_date_animation'] : '';
                    $date_animation_class = (!empty($date_animation)) ? 'mwt-posts-accordion-image-item-date-animation' : '';
                    $date_animation_duration = (!empty($settings['results_section_date_animation_duration'])) ? $settings['results_section_date_animation_duration'] : '';
                    $content_animation = (!empty($settings['results_section_content_animation'])) ? $settings['results_section_content_animation'] : '';
                    $content_animation_class = (!empty($content_animation)) ? 'mwt-posts-accordion-image-item-content-animation' : '';
                    $content_animation_duration = (!empty($settings['results_section_content_animation_duration'])) ? $settings['results_section_content_animation_duration'] : '';
                ?>
                    <div class="mwt-posts-accordion-image-item <?php echo (0 === $post_count) ? 'active' : '' ?>" id="mwt-posts-accordion-image-item-<?php echo esc_attr($unique_id) . '-' . $post_count ?>">
                        <?php if ($settings['show_date'] === 'yes') :
                        ?>
                            <a class="mwt-posts-accordion-image-item-date <?php echo esc_attr($date_animation_duration . ' ' . $date_animation_class); ?>" data-animationdelay="<?php echo esc_attr($animation_delay); ?>" data-dateanimation="<?php echo esc_attr($date_animation); ?>" href="<?php the_permalink($mwt_posts->ID); ?>">
                                <span><?php echo get_the_date('d', $mwt_posts->ID); ?></span>
                                <span><?php echo get_the_date('M', $mwt_posts->ID); ?></span>
                            </a>
                        <?php endif;
                        $settings['thumbnail'] = array(
                            'id' => get_post_thumbnail_id(),
                        );
                        $thumbnail_html = \Elementor\Group_Control_Image_Size::get_attachment_image_html($settings, 'thumbnail');
                        if (empty($thumbnail_html)) {
                            return;
                        }
                        echo wp_kses_post($thumbnail_html);

                        $mwt_post_title = get_the_title($mwt_posts->ID);
                        $mwt_title      = (!empty($mwt_post_title)) ? $mwt_post_title : esc_html__('(no title)', 'mwt-addons-for-elementor');
                        ?>
                        <div class="mwt-posts-accordion-image-item-content <?php echo esc_attr($content_animation_duration . ' ' . $content_animation_class); ?>" data-animationdelay="<?php echo esc_attr($animation_delay); ?>" data-contentanimation="<?php echo esc_attr($content_animation); ?>">
                            <h4 class="mwt-posts-accordion-image-item-title">
                                <a href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo wp_kses_post($mwt_title); ?></a>
                            </h4>
                            <?php
                            if ($settings['show_author'] === 'yes' || $settings['show_comments'] === 'yes' || $settings['show_likes'] === 'yes' || $settings['show_views'] === 'yes') : ?>
                                <footer class="mwt-posts-accordion-image-item-footer">
                                    <?php if ($settings['show_author'] === 'yes') :
                                        $author_icon = isset($settings['author_icon']) ? $settings['author_icon'] : ''; ?>
                                        <a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))) ?>">
                                            <?php
                                            if ($author_icon !== '') : ?>
                                                <?php \Elementor\Icons_Manager::render_icon($settings['author_icon'], ['aria-hidden' => 'true']); ?>
                                            <?php endif; ?>
                                            <?php esc_html_e('by ', 'mwt-addons-for-elementor'); ?><?php echo get_the_author() ?>
                                        </a>
                                    <?php endif;
                                    if ($settings['show_comments'] === 'yes' || $settings['show_likes'] === 'yes' || $settings['show_views'] === 'yes') : ?>
                                        <span class="mwt-posts-accordion-image-item-footer-end-block">
                                            <?php if ($settings['show_comments'] === 'yes') :
                                                $comments_icon = isset($settings['comments_icon']) ? $settings['comments_icon'] : '';
                                                if (comments_open()) : ?>
                                                    <span>
                                                        <a href="<?php comments_link(); ?>">
                                                            <?php
                                                            if ($comments_icon !== '') : ?>
                                                                <?php \Elementor\Icons_Manager::render_icon($settings['comments_icon'], ['aria-hidden' => 'true']); ?>
                                                            <?php endif; ?>
                                                            <span class="mwt-posts-accordion-image-item-footer-value"><?php comments_number("0", "1", "%"); ?></span>
                                                        </a>
                                                    </span>
                                                <?php else : ?>
                                                    <span>
                                                        <?php
                                                        if ($comments_icon !== '') : ?>
                                                            <?php \Elementor\Icons_Manager::render_icon($settings['comments_icon'], ['aria-hidden' => 'true']); ?>
                                                        <?php endif; ?>
                                                        <span><?php echo esc_html__('Closed', 'mwt-addons-for-elementor'); ?></span>
                                                    </span>
                                                <?php
                                                endif;
                                            endif;
                                            if (function_exists('mwt_get_post_likes') && $settings['show_likes'] === 'yes') : ?>
                                                <span>
                                                    <?php
                                                    $likes_icon = isset($settings['likes_icon']) ? $settings['likes_icon'] : '';
                                                    if ($likes_icon !== '') : ?>
                                                        <?php \Elementor\Icons_Manager::render_icon($settings['likes_icon'], ['aria-hidden' => 'true']); ?>
                                                    <?php endif; ?>
                                                    <span class="mwt-posts-accordion-image-item-footer-value"><?php echo mwt_get_post_likes(get_the_ID()); ?></span>
                                                </span>
                                            <?php
                                            endif;
                                            if (function_exists('mwt_get_post_views') && $settings['show_views'] === 'yes') : ?>
                                                <span>
                                                    <?php
                                                    $views_icon = isset($settings['views_icon']) ? $settings['views_icon'] : '';
                                                    if ($views_icon !== '') : ?>
                                                        <?php \Elementor\Icons_Manager::render_icon($settings['views_icon'], ['aria-hidden' => 'true']); ?>
                                                    <?php endif; ?>
                                                    <span class="mwt-posts-accordion-image-item-footer-value"><?php echo mwt_get_post_views(get_the_ID()); ?></span>
                                                </span>
                                            <?php endif; ?>
                                        </span>
                                    <?php endif; ?>
                                </footer>
                            <?php endif; ?>
                        </div>
                    </div><!-- .mwt-posts-accordion-image-item -->
                <?php
                    $post_count++;
                endwhile; ?>
                <?php wp_reset_postdata(); // reset the query 
                ?>
            </div>

            <div class="mwt-posts-accordion-nav">
                <?php
                $post_count = 0;
                while ($mwt_posts->have_posts()) :
                    $mwt_posts->the_post();
                ?>
                    <div class="mwt-posts-accordion-nav-item <?php echo (0 === $post_count) ? 'active' : '' ?>">
                        <div class="mwt-posts-accordion-nav-item-top-wrap">
                            <?php if ($settings['show_date'] === 'yes') : ?>
                                <a class="mwt-posts-accordion-nav-item-date" href="<?php the_permalink($mwt_posts->ID); ?>">
                                    <span><?php echo get_the_date('d', $mwt_posts->ID); ?></span>
                                    <span><?php echo get_the_date('M', $mwt_posts->ID); ?></span>
                                </a>
                            <?php endif;
                            $mwt_post_title = get_the_title($mwt_posts->ID);
                            $mwt_title      = (!empty($mwt_post_title)) ? $mwt_post_title : esc_html__('(no title)', 'mwt-addons-for-elementor');
                            ?>
                            <div class="mwt-posts-accordion-nav-item-content">
                                <h4 class="mwt-posts-accordion-nav-item-title">
                                    <?php echo wp_kses_post($mwt_title); ?>
                                </h4>
                                <?php
                                if ($settings['show_author'] === 'yes' || $settings['show_comments'] === 'yes' || $settings['show_likes'] === 'yes' || $settings['show_views'] === 'yes') : ?>
                                    <footer class="mwt-posts-accordion-nav-item-footer">
                                        <?php if ($settings['show_author'] === 'yes') :
                                            $author_icon = isset($settings['author_icon']) ? $settings['author_icon'] : ''; ?>
                                            <a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))) ?>">
                                                <?php
                                                if ($author_icon !== '') : ?>
                                                    <?php \Elementor\Icons_Manager::render_icon($settings['author_icon'], ['aria-hidden' => 'true']); ?>
                                                <?php endif; ?>
                                                <?php esc_html_e('by ', 'mwt-addons-for-elementor'); ?><?php echo get_the_author() ?>
                                            </a>
                                        <?php endif;
                                        if ($settings['show_comments'] === 'yes' || $settings['show_likes'] === 'yes' || $settings['show_views'] === 'yes') : ?>
                                            <span class="mwt-posts-accordion-nav-item-footer-end-block">
                                                <?php if ($settings['show_comments'] === 'yes') :
                                                    $comments_icon = isset($settings['comments_icon']) ? $settings['comments_icon'] : '';
                                                    if (comments_open()) : ?>
                                                        <span>
                                                            <a href="<?php comments_link(); ?>">
                                                                <?php
                                                                if ($comments_icon !== '') : ?>
                                                                    <?php \Elementor\Icons_Manager::render_icon($settings['comments_icon'], ['aria-hidden' => 'true']); ?>
                                                                <?php endif; ?>
                                                                <span class="mwt-posts-accordion-nav-item-footer-value"><?php comments_number("0", "1", "%"); ?></span>
                                                            </a>
                                                        </span>
                                                    <?php else : ?>
                                                        <span>
                                                            <?php
                                                            if ($comments_icon !== '') : ?>
                                                                <?php \Elementor\Icons_Manager::render_icon($settings['comments_icon'], ['aria-hidden' => 'true']); ?>
                                                            <?php endif; ?>
                                                            <span><?php echo esc_html__('Closed', 'mwt-addons-for-elementor'); ?></span>
                                                        </span>
                                                    <?php
                                                    endif;
                                                endif;
                                                if (function_exists('mwt_get_post_likes') && $settings['show_likes'] === 'yes') : ?>
                                                    <span>
                                                        <?php
                                                        $likes_icon = isset($settings['likes_icon']) ? $settings['likes_icon'] : '';
                                                        if ($likes_icon !== '') : ?>
                                                            <?php \Elementor\Icons_Manager::render_icon($settings['likes_icon'], ['aria-hidden' => 'true']); ?>
                                                        <?php endif; ?>
                                                        <span class="mwt-posts-accordion-nav-item-footer-value"><?php echo mwt_get_post_likes(get_the_ID()); ?></span>
                                                    </span>
                                                <?php
                                                endif;
                                                if (function_exists('mwt_get_post_views') && $settings['show_views'] === 'yes') : ?>
                                                    <span>
                                                        <?php
                                                        $views_icon = isset($settings['views_icon']) ? $settings['views_icon'] : '';
                                                        if ($views_icon !== '') : ?>
                                                            <?php \Elementor\Icons_Manager::render_icon($settings['views_icon'], ['aria-hidden' => 'true']); ?>
                                                        <?php endif; ?>
                                                        <span class="mwt-posts-accordion-nav-item-footer-value"><?php echo mwt_get_post_views(get_the_ID()); ?></span>
                                                    </span>
                                                <?php endif; ?>
                                            </span>
                                        <?php endif; ?>
                                    </footer>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="mwt-posts-accordion-nav-item-excerpt">
                            <div>
                                <?php
                                $length = $settings['posts_excerpt_length'];
                                echo wp_kses_post(wp_trim_words(get_the_excerpt($mwt_posts->ID), $length));
                                ?>
                            </div>
                        </div>
                    </div>
                <?php
                    $post_count++;
                endwhile; ?>
                <?php wp_reset_postdata(); // reset the query 
                ?>
            </div><!-- .mwt-posts-accordion-nav -->
        </div>
<?php
    }
} ?>