<?php

namespace MWT_Addons;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Plugin class.
 *
 * The main class that initiates and runs the addon.
 *
 * @since 1.0.0
 */
final class Plugin
{

    /**
     * Addon Version
     *
     * @since 1.0.0
     * @var string The addon version.
     */
    const VERSION = '1.0.1';

    /**
     * Minimum Elementor Version
     *
     * @since 1.0.0
     * @var string Minimum Elementor version required to run the addon.
     */
    const MINIMUM_ELEMENTOR_VERSION = '3.7.0';

    /**
     * Minimum PHP Version
     *
     * @since 1.0.0
     * @var string Minimum PHP version required to run the addon.
     */
    const MINIMUM_PHP_VERSION = '7.3';

    /**
     * Instance
     *
     * @since 1.0.0
     * @access private
     * @static
     * @var \MWT_Addons\Plugin The single instance of the class.
     */
    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @since 1.0.0
     * @access public
     * @static
     * @return \MWT_Addons\Plugin An instance of the class.
     */
    public static function instance()
    {

        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor
     *
     * Perform some compatibility checks to make sure basic requirements are meet.
     * If all compatibility checks pass, initialize the functionality.
     *
     * @since 1.0.0
     * @access public
     */
    public function __construct()
    {

        if ($this->is_compatible()) {
            add_action('elementor/init', [$this, 'init']);

            //extend existing widgets
            require_once(__DIR__ . '/extend.php');
        }
    }

    /**
     * Compatibility Checks
     *
     * Checks whether the site meets the addon requirement.
     *
     * @since 1.0.0
     * @access public
     */
    public function is_compatible()
    {

        // Check if Elementor installed and activated
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
            return false;
        }

        // Check for required Elementor version
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return false;
        }

        // Check for required PHP version
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return false;
        }

        return true;
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_missing_main_plugin()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor */
            esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'mwt-addons-for-elementor') . '</strong>'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required Elementor version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_elementor_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'mwt-addons-for-elementor') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_php_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('PHP', 'mwt-addons-for-elementor') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Initialize
     *
     * Load the addons functionality only after Elementor is initialized.
     *
     * Fired by `elementor/init` action hook.
     *
     * @since 1.0.0
     * @access public
     */
    public function init()
    {
        require_once(__DIR__ . '/functions.php');
        add_action('elementor/widgets/register', [$this, 'register_widgets']);
        add_action('elementor/frontend/after_enqueue_styles', [$this, 'frontend_styles']);
        add_action('elementor/frontend/after_enqueue_scripts', [$this, 'frontend_scripts']);
        add_action('elementor/elements/categories_registered', [$this, 'mwt_elementor_categories']);
    }

    public function mwt_elementor_categories($elements_manager)
    {
        // If the category is empty, it'll be automatically hidden by Elementor.
        $elements_manager->add_category(
            'mwt',
            array(
                'title' => __('MWT addons', 'mwt-addons-for-elementor'),
                'icon'   => 'fa fa-plug',
                'active' => false,
            )
        );
    }

    /**
     * Register Widgets
     *
     * Load widgets files and register new Elementor widgets.
     *
     * Fired by `elementor/widgets/register` action hook.
     *
     * @param \Elementor\Widgets_Manager $widgets_manager Elementor widgets manager.
     */
    public function register_widgets($widgets_manager)
    {
        require_once(__DIR__ . '/widgets/mwt-before-after.php');
        require_once(__DIR__ . '/widgets/mwt-countdown.php');
        require_once(__DIR__ . '/widgets/mwt-counter-up.php');
        require_once(__DIR__ . '/widgets/mwt-gallery-carousel.php');
        require_once(__DIR__ . '/widgets/mwt-posts-accordion.php');
        require_once(__DIR__ . '/widgets/mwt-testimonials-carousel.php');
        require_once(__DIR__ . '/widgets/mwt-main-slider.php');
        require_once(__DIR__ . '/widgets/mwt-gallery-grid.php');
        require_once(__DIR__ . '/widgets/mwt-services-grid.php');
        require_once(__DIR__ . '/widgets/mwt-products-carousel.php');
        require_once(__DIR__ . '/widgets/mwt-team.php');
        require_once(__DIR__ . '/widgets/mwt-posts.php');
        require_once(__DIR__ . '/widgets/mwt-tabs-accordion.php');
        require_once(__DIR__ . '/widgets/mwt-parallax-box.php');

        $widgets_manager->register(new \Elementor_MWT_Before_After());
        $widgets_manager->register(new \Elementor_MWT_Countdown());
        $widgets_manager->register(new \Elementor_MWT_Counter_Up());
        $widgets_manager->register(new \Elementor_MWT_Gallery_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Posts_Accordion());
        $widgets_manager->register(new \Elementor_MWT_Testimonials_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Main_Slider());
        $widgets_manager->register(new \Elementor_MWT_Gallery_Grid());
        $widgets_manager->register(new \Elementor_MWT_Services_Grid());
        $widgets_manager->register(new \Elementor_MWT_Products_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Team());
        $widgets_manager->register(new \Elementor_MWT_Posts());
        $widgets_manager->register(new \Elementor_MWT_Tabs_Accordion());
        $widgets_manager->register(new \Elementor_MWT_Parallax_Box());
    }



    public function frontend_styles()
    {
        wp_register_style('owl-carousel-style', plugins_url('../assets/css/owl-carousel.css', __FILE__));
        wp_register_style('mwt-before-after-style', plugins_url('../assets/css/mwt-before-after.css', __FILE__));
        wp_register_style('mwt-countdown-style', plugins_url('../assets/css/mwt-countdown.css', __FILE__));
        wp_register_style('mwt-counter-up-style', plugins_url('../assets/css/mwt-counter-up.css', __FILE__));
        wp_register_style('mwt-gallery-carousel-style', plugins_url('../assets/css/mwt-gallery-carousel.css', __FILE__));
        wp_register_style('mwt-posts-accordion-style', plugins_url('../assets/css/mwt-posts-accordion.css', __FILE__));
        wp_register_style('mwt-testimonials-carousel-style', plugins_url('../assets/css/mwt-testimonials-carousel.css', __FILE__));
        wp_register_style('mwt-main-slider-style', plugins_url('../assets/css/mwt-main-slider.css', __FILE__));
        wp_register_style('mwt-gallery-grid-style', plugins_url('../assets/css/mwt-gallery-grid.css', __FILE__));
        wp_register_style('mwt-services-grid-style', plugins_url('../assets/css/mwt-services-grid.css', __FILE__));
        wp_register_style('mwt-products-carousel-style', plugins_url('../assets/css/mwt-products-carousel.css', __FILE__));
        wp_register_style('mwt-team-style', plugins_url('../assets/css/mwt-team.css', __FILE__));
        wp_register_style('mwt-posts-style', plugins_url('../assets/css/mwt-posts.css', __FILE__));
        wp_register_style('mwt-tabs-accordion-style', plugins_url('../assets/css/mwt-tabs-accordion.css', __FILE__));
        wp_register_style('mwt-parallax-box-style', plugins_url('../assets/css/mwt-parallax-box.css', __FILE__));

        wp_enqueue_style('owl-carousel-style');
        wp_enqueue_style('mwt-before-after-style');
        wp_enqueue_style('mwt-countdown-style');
        wp_enqueue_style('mwt-counter-up-style');
        wp_enqueue_style('mwt-gallery-carousel-style');
        wp_enqueue_style('mwt-posts-accordion-style');
        wp_enqueue_style('mwt-testimonials-carousel-style');
        wp_enqueue_style('mwt-main-slider-style');
        wp_enqueue_style('mwt-gallery-grid-style');
        wp_enqueue_style('mwt-services-grid-style');
        wp_enqueue_style('mwt-products-carousel-style');
        wp_enqueue_style('mwt-team-style');
        wp_enqueue_style('mwt-posts-style');
        wp_enqueue_style('mwt-tabs-accordion-style');
        wp_enqueue_style('mwt-parallax-box-style');
    }

    public function frontend_scripts()
    {
        wp_register_script('mwt-before-after-script', plugins_url('../assets/js/mwt-before-after.js', __FILE__), ['jquery']);
        wp_register_script('jquery-countdown-script', plugins_url('../assets/js/jquery.countdown.min.js', __FILE__), ['jquery']);
        wp_register_script('mwt-countdown-script', plugins_url('../assets/js/mwt-countdown.js', __FILE__), ['jquery']);
        wp_register_script('mwt-counter-up-script', plugins_url('../assets/js/mwt-counter-up.js', __FILE__), ['jquery']);
        wp_register_script('mwt-gallery-carousel-script', plugins_url('../assets/js/mwt-gallery-carousel.js', __FILE__), ['jquery']);
        wp_register_script('mwt-posts-accordion-script', plugins_url('../assets/js/mwt-posts-accordion.js', __FILE__), ['jquery']);
        wp_register_script('mwt-testimonials-carousel-script', plugins_url('../assets/js/mwt-testimonials-carousel.js', __FILE__), ['jquery']);
        wp_register_script('mwt-main-slider-script', plugins_url('../assets/js/mwt-main-slider.js', __FILE__), ['jquery']);
        wp_register_script('mwt-gallery-grid-script', plugins_url('../assets/js/mwt-gallery-grid.js', __FILE__), ['jquery']);
        wp_register_script('mwt-services-grid-script', plugins_url('../assets/js/mwt-services-grid.js', __FILE__), ['jquery']);
        wp_register_script('mwt-products-carousel-script', plugins_url('../assets/js/mwt-products-carousel.js', __FILE__), ['jquery']);
        wp_register_script('mwt-team-script', plugins_url('../assets/js/mwt-team.js', __FILE__), ['jquery']);
        wp_register_script('mwt-posts-script', plugins_url('../assets/js/mwt-posts.js', __FILE__), ['jquery']);
        wp_register_script('mwt-tabs-accordion-script', plugins_url('../assets/js/mwt-tabs-accordion.js', __FILE__), ['jquery']);

        wp_register_script('parallax-script', plugins_url('../assets/js/parallax.min.js', __FILE__), ['jquery']);
        wp_register_script('mwt-parallax-box-script', plugins_url('../assets/js/mwt-parallax-box.js', __FILE__), ['jquery']);

        wp_enqueue_script('mwt-before-after-script');
        wp_enqueue_script('jquery-countdown-script');
        wp_enqueue_script('mwt-countdown-script');
        wp_enqueue_script('mwt-counter-up-script');
        wp_enqueue_script('mwt-gallery-carousel-script');
        wp_enqueue_script('mwt-posts-accordion-script');
        wp_enqueue_script('mwt-testimonials-carousel-script');
        wp_enqueue_script('mwt-main-slider-script');
        wp_enqueue_script('mwt-gallery-grid-script');
        wp_enqueue_script('mwt-services-grid-script');
        wp_enqueue_script('mwt-products-carousel-script');
        wp_enqueue_script('mwt-team-script');
        wp_enqueue_script('mwt-tabs-accordion-script');
        wp_enqueue_script('mwt-tabs-accordion-script');

        wp_enqueue_script('parallax-script');
        wp_enqueue_script('mwt-parallax-box-script');
    }
}
